require_relative "rubysermon/version"
require_relative "rubysermon/mod_loader"
require_relative "rubysermon/mod_template"
require_relative "rubysermon/configurator"
require_relative "rubysermon/db"

require "date"

module Rubysermon
	LIB_PATH = File.expand_path(File.dirname(__FILE__))
	APP_PATH = "#{LIB_PATH}/.."
	MOD_PATH = "#{LIB_PATH}/rubysermon/mod"

	class App
		def initialize
			@config 		= { repeat: 60,
								config_path: "#{LIB_PATH}/rubysermon/config.json",
								db_path: "/tmp/rubysermon_history.db"
			}

			@running_mods	= []
			@enabled_mods	= []
			@db				= nil

			check_args()
		end

		def run
			read_config()
			enable_and_start_mods()

			if (msg = cannot_start_fetch_process_sleep_cycle?)
				abort(msg)
			end

			#start_db()
			start_fetch_process_sleep_cycle()
		end

		private

		def check_args
			if not ARGV.empty?
				config_path = ARGV.first
				if File.exists?(config_path)
					@config[:config_path] = config_path
				else
					abort("Wrong config file path. Exiting.")
				end
			end
		end

		def read_config
			configurator = Configurator.new(@config[:config_path])
			config = configurator.get_settings()
			@config.merge!(config)
		end

		def enable_and_start_mods
			@enabled_mods = @config[:modules].to_a
			@enabled_mods.each do |mod| load_mod(mod) end
		end

		def load_mod(mod_name)
			begin
				mod = ModLoader.load(mod_name)
				@running_mods.push(mod)
			rescue ModLoaderException => e
				$stderr.puts e.message
			end
		end

		def cannot_start_fetch_process_sleep_cycle?
			if @config[:repeat].to_i < 1
				return "Repeat cycle is too short"
			elsif @running_mods.empty?
				return "There are no enabled modules"
			end

			false
		end

		def start_db
			@db = DB.new(@config[:db_path])
		end

		def start_fetch_process_sleep_cycle
			while true
				current_time = DateTime.now()
				@running_mods.each do |mod|
					result = mod.results
					notify = mod.notify?
				end
				sleep @config[:repeat]
			end
		end
	end
end
